var BASE_URL = 'https://hooz.pages.dev/';

var domainInput = document.getElementById('domain-input');
var checkBtn    = document.getElementById('check-btn');
var errorMsg    = document.getElementById('error-msg');
var tabStrip    = document.getElementById('tab-strip');
var tabDomain   = document.getElementById('tab-domain');

// ── IDN / Arabic / Punycode support ──────────
// The browser's URL API converts Unicode → Punycode in .hostname
// but stores the original Unicode in .host if available.
// We use this to:
//  1. Validate IDN domains (convert to punycode, check if valid ASCII)
//  2. Display the human-readable Unicode form to the user
//  3. Pass the Unicode form to hooz.pages.dev (it handles conversion itself)

function toASCIIDomain(domain) {
  // Converts الرياضيات.com → xn--mgbcd4a2b3c.com via browser URL API
  try {
    return new URL('http://' + domain).hostname;
  } catch(e) {
    return domain;
  }
}

function toUnicodeDomain(domain) {
  // Converts xn--mgbcd4a2b3c.com back to الرياضيات.com for display
  try {
    // URLSearchParams trick: use the browser's built-in IDN decoder
    // new URL handles xn-- → unicode automatically in some contexts,
    // but the safest cross-browser way is just to pass it through:
    var ascii = new URL('http://' + domain).hostname;
    // If the browser decoded it to unicode already, great; otherwise keep ascii
    return ascii;
  } catch(e) {
    return domain;
  }
}

// ── Helpers ───────────────────────────────────
function cleanDomain(str) {
  // Do NOT call toLowerCase — it corrupts Arabic characters
  return str
    .replace(/^(https?:\/\/)?(www\.)?/i, '')
    .split('/')[0]
    .split('?')[0]
    .trim();
}

// Accepts ASCII and Unicode/IDN domains (Arabic, CJK, Devanagari, etc.)
function isValidDomain(d) {
  if (!d || d.length > 253) return false;
  if (/\s/.test(d)) return false;
  if (d.indexOf('.') === -1) return false;

  // Convert to punycode — if result is a valid ASCII hostname, accept it
  var ascii = toASCIIDomain(d);
  if (/^[a-zA-Z0-9]([a-zA-Z0-9\-]{0,61}[a-zA-Z0-9])?(\.[a-zA-Z0-9\-\.]{1,}[a-zA-Z]{2,})$/.test(ascii)) {
    return true;
  }

  // Fallback: accept raw Unicode IDN if it has ≥2 dot-separated parts,
  // TLD ≥2 chars, and no empty labels
  var parts = d.split('.');
  if (parts.length < 2) return false;
  if (parts[parts.length - 1].length < 2) return false;
  if (parts.some(function(p) { return p.length === 0; })) return false;
  return true;
}

function showError(msg) {
  errorMsg.textContent = '⚠ ' + msg;
  errorMsg.classList.add('show');
}

function hideError() {
  errorMsg.classList.remove('show');
}

function openWhois(domain) {
  // Pass the domain as-is (Unicode or ASCII) — hooz.pages.dev handles both
  var url = BASE_URL + '?domain=' + encodeURIComponent(domain);
  chrome.tabs.create({ url: url });
  window.close();
}

// ── Check button ──────────────────────────────
checkBtn.addEventListener('click', function() {
  hideError();
  var raw = domainInput.value.trim();  // no toLowerCase — preserve Arabic
  if (!raw) { showError('Enter a domain name.'); return; }
  var domain = cleanDomain(raw);
  if (!isValidDomain(domain)) { showError('"' + domain + '" is not a valid domain.'); return; }
  openWhois(domain);
});

domainInput.addEventListener('keydown', function(e) {
  if (e.key === 'Enter') checkBtn.click();
});

// Focus input on open
domainInput.focus();

// ── Quick TLD buttons ─────────────────────────
// Type a name (Arabic or Latin), click a TLD to instantly check that variant
document.querySelectorAll('.quick-btn').forEach(function(btn) {
  btn.addEventListener('click', function() {
    hideError();
    var tld = btn.getAttribute('data-tld');
    var current = domainInput.value.trim();  // preserve Arabic
    if (!current) { showError('Type a domain name first.'); return; }

    // Strip any existing TLD suffix and append the chosen one
    var cleaned = cleanDomain(current);
    var base = cleaned.replace(/\.[^.]+$/, '');
    if (!base) { showError('Type a domain name first.'); return; }

    var domain = base + tld;
    if (!isValidDomain(domain)) { showError('"' + domain + '" is not a valid domain.'); return; }
    openWhois(domain);
  });
});

// ── Current tab detection ─────────────────────
chrome.tabs.query({ active: true, currentWindow: true }, function(tabs) {
  if (!tabs || !tabs[0] || !tabs[0].url) {
    tabDomain.textContent = 'No active tab';
    tabStrip.style.opacity = '0.4';
    tabStrip.style.pointerEvents = 'none';
    return;
  }

  var url = tabs[0].url;

  // Skip browser-internal pages
  if (url.startsWith('chrome://') || url.startsWith('chrome-extension://') ||
      url.startsWith('about:') || url.startsWith('edge://')) {
    tabDomain.textContent = 'Browser page — not checkable';
    tabStrip.style.opacity = '0.4';
    tabStrip.style.pointerEvents = 'none';
    return;
  }

  try {
    var parsed = new URL(url);

    // parsed.hostname gives punycode for IDN sites (e.g. xn--mgbcd4a2b3c.com)
    // Use the original URL string to extract the unicode hostname for display
    var rawHostname = parsed.hostname;           // may be xn-- punycode
    var displayHost = rawHostname.replace(/^www\./, '');

    // Try to get the unicode version from the original URL for nicer display
    // (Chrome sometimes gives us the unicode version directly in the tab URL)
    try {
      var urlObj = new URL(url);
      // If the URL contains percent-encoded unicode, decode it
      var decoded = decodeURIComponent(urlObj.hostname);
      if (decoded !== urlObj.hostname) {
        displayHost = decoded.replace(/^www\./, '');
      }
    } catch(e2) { /* keep displayHost as-is */ }

    tabDomain.textContent = displayHost;

    tabStrip.addEventListener('click', function() {
      // Use the original hostname (could be punycode) — hooz.pages.dev handles it
      var checkDomain = rawHostname.replace(/^www\./, '');
      if (isValidDomain(checkDomain)) {
        openWhois(checkDomain);
      } else {
        showError('Cannot check "' + checkDomain + '"');
      }
    });

  } catch(e) {
    tabDomain.textContent = 'Cannot parse URL';
    tabStrip.style.opacity = '0.4';
    tabStrip.style.pointerEvents = 'none';
  }
});
