var BASE_URL = 'https://hooz.pages.dev/';

// ── Create context menu on install ────────────
chrome.runtime.onInstalled.addListener(function() {
  chrome.contextMenus.create({
    id: 'whois-lookup',
    title: '🔍 WHOIS Lookup: "%s"',
    contexts: ['selection']   // only shows when text is selected
  });
});

// ── Handle right-click → WHOIS ─────────────────
chrome.contextMenus.onClicked.addListener(function(info, tab) {
  if (info.menuItemId !== 'whois-lookup') return;

  var selected = (info.selectionText || '').trim();
  if (!selected) return;

  // Clean the selected text into a domain
  var domain = cleanDomain(selected);

  if (!isValidDomain(domain)) {
    // If the raw selection isn't a domain, try extracting one from a URL
    domain = extractDomainFromURL(selected);
  }

  if (!isValidDomain(domain)) {
    // Nothing usable — open the checker with the raw text pre-filled so
    // the user can see the error message rather than nothing happening
    domain = selected.replace(/\s+/g, '').substring(0, 253);
  }

  var url = BASE_URL + '?domain=' + encodeURIComponent(domain);
  chrome.tabs.create({ url: url });
});

// ── Helpers ───────────────────────────────────

function toASCIIDomain(domain) {
  try {
    return new URL('http://' + domain).hostname;
  } catch(e) {
    return domain;
  }
}

function cleanDomain(str) {
  return str
    .replace(/^(https?:\/\/)?(www\.)?/i, '')
    .split('/')[0]
    .split('?')[0]
    .trim();
}

// Try to pull a domain out of a full URL string the user may have selected
function extractDomainFromURL(str) {
  try {
    var parsed = new URL(str.indexOf('://') === -1 ? 'http://' + str : str);
    return parsed.hostname.replace(/^www\./, '');
  } catch(e) {
    return str;
  }
}

function isValidDomain(d) {
  if (!d || d.length > 253) return false;
  if (/\s/.test(d)) return false;
  if (d.indexOf('.') === -1) return false;

  var ascii = toASCIIDomain(d);
  if (/^[a-zA-Z0-9]([a-zA-Z0-9\-]{0,61}[a-zA-Z0-9])?(\.[a-zA-Z0-9\-\.]{1,}[a-zA-Z]{2,})$/.test(ascii)) {
    return true;
  }

  // Accept raw Unicode IDN (Arabic etc.) if structure looks valid
  var parts = d.split('.');
  if (parts.length < 2) return false;
  if (parts[parts.length - 1].length < 2) return false;
  if (parts.some(function(p) { return p.length === 0; })) return false;
  return true;
}
